<?php
declare(strict_types=1);

date_default_timezone_set('Europe/London');

// --------------------------------------------------
// CONFIG
// --------------------------------------------------

// Database connection
define('DB_HOST', 'localhost:3306');
define('DB_NAME', 'radiodj-database');
define('DB_USER', 'radiodj-user');
define('DB_PASS', 'radiodj-password');

// Station / stream details
define('STATION_NAME',  'Artwave Radio');
define('STREAM_URL',    'https://uk2-pn.mixstream.net/PORT/stream'); // <- your stream URL

// List size
define('SONGS_PER_PAGE', 18);

// Requests system
define('REQUESTS_ALLOW', 'y'); // 'y' or 'n'

// Album art
define('ALBUM_ART_DIR', __DIR__ . '/Album-Art');   // filesystem
define('ALBUM_ART_URL', 'Album-Art');              // web path
define('DEFAULT_ART_URL', 'assets/imgs/noart.png'); // change to your default image

// --------------------------------------------------
// PDO CONNECTION ($pdo_scs)
// --------------------------------------------------
$dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';

try {
    $pdo_scs = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo "Database connection failed.";
    exit;
}

// --------------------------------------------------
// HELPER: CURL GET
// --------------------------------------------------
function curlGet(string $url): ?string {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT      => "OneRadio/1.0"
    ]);
    $out = curl_exec($ch);
    curl_close($ch);
    return $out !== false ? $out : null;
}

// --------------------------------------------------
// HELPER: slug / safe filename
// --------------------------------------------------
function slugify(string $str): string {
    $str = trim($str);
    if ($str === '') {
        return 'art';
    }
    $str = iconv('UTF-8', 'ASCII//TRANSLIT', $str);
    $str = preg_replace('~[^a-zA-Z0-9]+~', '-', $str);
    $str = trim($str, '-');
    $str = strtolower($str);
    return $str !== '' ? $str : 'art';
}

// --------------------------------------------------
// HELPER: find local album art
// --------------------------------------------------
function findLocalArt(string $artist, string $title): ?string {
    $artistSlug = slugify($artist);
    $titleSlug  = slugify($title);

    $candidates = [];
    if ($artistSlug && $titleSlug) {
        $candidates[] = $artistSlug . '-' . $titleSlug;
    }
    if ($titleSlug) {
        $candidates[] = $titleSlug;
    }
    if ($artistSlug) {
        $candidates[] = $artistSlug;
    }

    $exts = ['jpg', 'jpeg', 'png', 'webp'];

    foreach ($candidates as $base) {
        foreach ($exts as $ext) {
            $fs = ALBUM_ART_DIR . '/' . $base . '.' . $ext;
            if (is_file($fs)) {
                return ALBUM_ART_URL . '/' . basename($fs);
            }
        }
    }
    return null;
}

// --------------------------------------------------
// HELPER: fetch & cache artwork from iTunes
// --------------------------------------------------
function fetchItunesArtToCache(string $artist, string $title): ?string {
    $term = trim($artist . ' ' . $title);
    if ($term === '') {
        return null;
    }

    $query = urlencode($term);
    $url   = "https://itunes.apple.com/search?term={$query}&entity=song&limit=1";

    $json = curlGet($url);
    if (!$json) {
        return null;
    }

    $data = json_decode($json, true);
    if (!isset($data['results'][0]['artworkUrl100'])) {
        return null;
    }

    $artUrl = $data['results'][0]['artworkUrl100'];
    // try to upgrade to 600x600
    $artUrl = str_replace('100x100bb', '600x600bb', $artUrl);

    $imgData = curlGet($artUrl);
    if (!$imgData) {
        // fallback: just use remote URL if download fails
        return $artUrl;
    }

    if (!is_dir(ALBUM_ART_DIR)) {
        @mkdir(ALBUM_ART_DIR, 0775, true);
    }

    $base = slugify($artist . '-' . $title);
    $fs   = ALBUM_ART_DIR . '/' . $base . '.jpg';

    if (@file_put_contents($fs, $imgData) !== false) {
        return ALBUM_ART_URL . '/' . basename($fs);
    }

    // if writing fails, just use remote URL
    return $artUrl;
}

// --------------------------------------------------
// HELPER: combined album art getter
// --------------------------------------------------
function getAlbumArtUrl(string $artist, string $title): string {
    // 1. local
    $local = findLocalArt($artist, $title);
    if ($local) return $local;

    // 2. iTunes (and cache)
    $remote = fetchItunesArtToCache($artist, $title);
    if ($remote) return $remote;

    // 3. default
    return DEFAULT_ART_URL;
}

// --------------------------------------------------
// AJAX: SONG LIST (GRID) - search + pagination
// --------------------------------------------------
if (isset($_GET['ajax']) && $_GET['ajax'] === 'songs') {
    $page    = max(1, (int)($_GET['page'] ?? 1));
    $perPage = SONGS_PER_PAGE;
    $offset  = ($page - 1) * $perPage;
    $q       = trim((string)($_GET['q'] ?? ''));

    $where  = '';
    $params = [];

    if ($q !== '') {
        $where = "WHERE (artist LIKE :q OR title LIKE :q OR album LIKE :q)";
        $params[':q'] = '%' . $q . '%';
    }

    // total
    $sqlCount = "SELECT COUNT(*) FROM songs {$where}";
    $stmtCount = $pdo_scs->prepare($sqlCount);
    if ($q !== '') {
        $stmtCount->bindValue(':q', '%' . $q . '%', PDO::PARAM_STR);
    }
    $stmtCount->execute();
    $total = (int)$stmtCount->fetchColumn();
    $totalPages = max(1, (int)ceil($total / $perPage));

    // rows
    $sql = "
        SELECT
            id,
            artist,
            title,
            album,
            duration
        FROM songs
        {$where}
        ORDER BY artist ASC, title ASC
        LIMIT :limit OFFSET :offset
    ";
    $stmt = $pdo_scs->prepare($sql);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v, PDO::PARAM_STR);
    }
    $stmt->bindValue(':limit',  $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset,  PDO::PARAM_INT);
    $stmt->execute();

    $songs = [];
    while ($row = $stmt->fetch()) {
        $artist = $row['artist'] ?? '';
        $title  = $row['title'] ?? '';

        $artUrl = getAlbumArtUrl($artist, $title);

        $songs[] = [
            'id'       => (int)$row['id'],
            'artist'   => $artist,
            'title'    => $title,
            'album'    => $row['album'] ?? '',
            'duration' => (int)($row['duration'] ?? 0),
            'likes'    => 0, // placeholder; wire to DB later if needed
            'art'      => $artUrl,
        ];
    }

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'songs'      => $songs,
        'page'       => $page,
        'totalPages' => $totalPages,
        'total'      => $total,
    ]);
    exit;
}

// --------------------------------------------------
// AJAX: REQUEST TRACK
// --------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'request') {
    header('Content-Type: application/json; charset=utf-8');

    if (REQUESTS_ALLOW !== 'y') {
        echo json_encode([
            'ok'    => false,
            'error' => 'Requests are currently disabled.',
        ]);
        exit;
    }

    $songId   = (int)($_POST['song_id'] ?? 0);
    $username = trim((string)($_POST['username'] ?? ''));
    $message  = trim((string)($_POST['message'] ?? ''));

    if ($songId < 1 || $username === '') {
        echo json_encode([
            'ok'    => false,
            'error' => 'Please enter your name to send a request.',
        ]);
        exit;
    }

    try {
        // ensure song exists
        $chk = $pdo_scs->prepare("SELECT id FROM songs WHERE id = :id LIMIT 1");
        $chk->execute([':id' => $songId]);
        if (!$chk->fetch()) {
            echo json_encode([
                'ok'    => false,
                'error' => 'Song not found.',
            ]);
            exit;
        }

        // Insert into your requests table
        // Adjust column names if needed
        $stmt = $pdo_scs->prepare("
            INSERT INTO requests
                (songID, username, message, userIP, requested)
            VALUES
                (:songID, :username, :message, :userIP, NOW())
        ");

        $stmt->execute([
            ':songID'       => $songId,
            ':username' => mb_substr($username, 0, 100),
            ':message'  => mb_substr($message, 0, 500),
            ':userIP'   => $_SERVER['REMOTE_ADDR'] ?? '',
        ]);

        echo json_encode([
            'ok'      => true,
            'message' => 'Your request has been sent to the playlist!',
        ]);
    } catch (Throwable $e) {
        echo json_encode([
            'ok'    => false,
            'error' => 'There was a problem saving your request. Please try again later.',
        ]);
    }

    exit;
}

// --------------------------------------------------
// MAIN PAGE
// --------------------------------------------------
$requestsOn = (REQUESTS_ALLOW === 'y');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Requests | <?php echo htmlspecialchars(STATION_NAME); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- Font Awesome -->
    <link rel="stylesheet"
          href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">

    <style>
        :root {
            --bg: #050914;
            --bg-alt: #0b101f;
            --bg-elevated: #121624;
            --accent: #ff4b6e;
            --accent-soft: #2b1b3f;
            --text: #f5f7ff;
            --text-muted: #9ba3c6;
            --border-subtle: #252a3d;
            --card-radius: 18px;
        }

        * { box-sizing: border-box; }

        body {
            margin: 0;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: radial-gradient(circle at top, #161d3b 0, var(--bg) 55%);
            color: var(--text);
        }

        .page {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }

        .header {
            position: sticky;
            top: 0;
            z-index: 20;
            backdrop-filter: blur(18px);
            background: linear-gradient(to right, rgba(5, 9, 20, 0.96), rgba(10, 12, 24, 0.96));
            border-bottom: 1px solid var(--border-subtle);
        }

        .header__inner {
            max-width: 1200px;
            margin: 0 auto;
            padding: 12px 18px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 18px;
        }

        .header__brand {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }

        .header__title {
            font-weight: 700;
            letter-spacing: .08em;
            font-size: 0.98rem;
            text-transform: uppercase;
        }

        .header__subtitle {
            font-size: 0.8rem;
            color: var(--text-muted);
        }

        .player {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 8px 12px;
            border-radius: 999px;
            background: radial-gradient(circle at top, var(--accent-soft), transparent 70%);
            border: 1px solid rgba(255, 75, 110, 0.4);
            box-shadow: 0 0 20px rgba(255, 75, 110, 0.35);
        }

        .player__btn {
            width: 34px;
            height: 34px;
            border-radius: 50%;
            border: none;
            cursor: pointer;
            background: var(--accent);
            color: #fff;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
        }

        .player__btn i { pointer-events: none; }

        .player__meta {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }

        .player__station {
            font-size: 0.8rem;
            font-weight: 600;
        }

        .player__status {
            font-size: 0.75rem;
            color: var(--text-muted);
        }

        .player__volume-wrap {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .player__volume-icon {
            font-size: 0.8rem;
            color: var(--text-muted);
        }

        .player__volume {
            width: 110px;
        }

        main { flex: 1; }

        .main-inner {
            max-width: 1200px;
            margin: 18px auto 30px;
            padding: 0 18px 24px;
        }

        .panel {
            border-radius: 22px;
            background: radial-gradient(circle at top left, #171f3c, var(--bg-elevated));
            border: 1px solid var(--border-subtle);
            box-shadow: 0 24px 60px rgba(0, 0, 0, 0.7);
            padding: 18px 18px 12px;
        }

        .panel__header {
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            align-items: center;
            gap: 14px;
            margin-bottom: 14px;
        }

        .panel__title {
            font-size: 1.05rem;
            font-weight: 600;
        }

        .panel__subtitle {
            font-size: 0.8rem;
            color: var(--text-muted);
        }

        .search {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 6px 12px;
            border-radius: 999px;
            background: #0c1222;
            border: 1px solid var(--border-subtle);
        }

        .search__icon {
            font-size: 0.9rem;
            color: var(--text-muted);
        }

        .search input {
            border: none;
            outline: none;
            background: transparent;
            color: var(--text);
            font-size: 0.86rem;
            width: 180px;
        }

        .search input::placeholder {
            color: var(--text-muted);
        }

        .songs-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(170px, 1fr));
            gap: 16px;
            margin-top: 4px;
        }

        .song-card {
            background: #101320;
            border-radius: var(--card-radius);
            overflow: hidden;
            display: flex;
            flex-direction: column;
            box-shadow: 0 14px 30px rgba(0, 0, 0, 0.7);
            border: 1px solid rgba(255, 255, 255, 0.03);
            position: relative;
        }

        .song-card__art {
            position: relative;
            padding-top: 100%; /* square */
            overflow: hidden;
        }

        .song-card__art img {
            position: absolute;
            inset: 0;
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.25s ease, filter 0.25s ease;
        }

        .song-card__body {
            padding: 10px 11px 8px;
            display: flex;
            flex-direction: column;
            gap: 3px;
        }

        .song-title {
            font-size: 0.9rem;
            font-weight: 600;
            white-space: nowrap;
            text-overflow: ellipsis;
            overflow: hidden;
        }

        .song-artist {
            font-size: 0.8rem;
            color: var(--text-muted);
            white-space: nowrap;
            text-overflow: ellipsis;
            overflow: hidden;
        }

        .song-likes {
            font-size: 0.75rem;
            color: var(--text-muted);
            margin-top: 2px;
        }

        .song-card__footer {
            display: flex;
            justify-content: flex-end;
            padding: 0 10px 9px;
        }

        .song-request-btn {
            border-radius: 999px;
            border: none;
            padding: 5px 11px;
            font-size: 0.78rem;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            background: linear-gradient(135deg, var(--accent), #ff8a77);
            color: #fff;
            box-shadow: 0 0 14px rgba(255, 138, 119, 0.45);
        }

        .song-request-btn i {
            font-size: 0.8rem;
        }

        .song-card:hover .song-card__art img {
            transform: scale(1.05);
            filter: brightness(1.05);
        }

        .song-card:hover {
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.85);
            border-color: rgba(255, 255, 255, 0.06);
        }

        .meta-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 10px;
            margin-top: 14px;
            font-size: 0.78rem;
            color: var(--text-muted);
        }

        .pagination {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .pagination button {
            border-radius: 999px;
            border: 1px solid var(--border-subtle);
            background: #101524;
            color: var(--text);
            font-size: 0.76rem;
            padding: 4px 10px;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 4px;
        }

        .pagination button[disabled] {
            opacity: .4;
            cursor: default;
        }

        .empty-grid {
            grid-column: 1 / -1;
            text-align: center;
            padding: 36px 10px;
            color: var(--text-muted);
            font-size: 0.86rem;
        }

        /* MODAL (Style 1: big artist/album header) */
        .modal-backdrop {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.8);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 40;
        }

        .modal-backdrop.is-visible {
            display: flex;
        }

        .modal {
            width: 100%;
            max-width: 480px;
            background: #0d1020;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 30px 80px rgba(0, 0, 0, 0.95);
            border: 1px solid rgba(255, 255, 255, 0.06);
        }

        .modal__hero {
            position: relative;
            height: 200px;
            background-size: cover;
            background-position: center;
        }

        .modal__hero::after {
            content: "";
            position: absolute;
            inset: 0;
            background: linear-gradient(to bottom, rgba(0,0,0,0.1), rgba(5,7,16,0.96));
        }

        .modal__hero-inner {
            position: absolute;
            inset: 0;
            padding: 14px 14px 10px;
            display: flex;
            flex-direction: column;
            justify-content: flex-end;
            gap: 4px;
        }

        .modal__hero-artist {
            font-size: 0.78rem;
            text-transform: uppercase;
            letter-spacing: .12em;
            opacity: 0.85;
        }

        .modal__hero-title {
            font-size: 1rem;
            font-weight: 600;
        }

        .modal__close-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            border-radius: 999px;
            border: none;
            padding: 4px 7px;
            background: rgba(0, 0, 0, 0.55);
            color: #fff;
            cursor: pointer;
        }

        .modal__body {
            padding: 12px 14px 14px;
        }

        .modal__sub {
            font-size: 0.8rem;
            color: var(--text-muted);
            margin-bottom: 9px;
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 8px;
            margin-bottom: 8px;
        }

        .field label {
            display: block;
            font-size: 0.75rem;
            margin-bottom: 3px;
            color: var(--text-muted);
        }

        .field input,
        .field textarea {
            width: 100%;
            border-radius: 9px;
            padding: 7px 8px;
            border: 1px solid var(--border-subtle);
            background: #10182b;
            color: var(--text);
            font-size: 0.8rem;
            resize: vertical;
        }

        .field textarea {
            min-height: 60px;
        }

        .modal__footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 6px;
        }

        .modal__hint {
            font-size: 0.72rem;
            color: var(--text-muted);
        }

        .btn--primary {
            background: linear-gradient(135deg, var(--accent), #ff8a77);
            color: #fff;
            border: none;
            border-radius: 999px;
            padding: 6px 14px;
            font-size: 0.8rem;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn--primary i {
            font-size: 0.8rem;
        }

        .alert {
            font-size: 0.78rem;
            margin-top: 4px;
        }

        .alert--error {
            color: #ffb3b3;
        }

        .alert--success {
            color: #a8ffcb;
        }

        @media (max-width: 640px) {
            .header__inner {
                flex-direction: column;
                align-items: flex-start;
            }
            .player {
                width: 100%;
                justify-content: space-between;
            }
            .player__volume { width: 90px; }
        }
    </style>
</head>
<body data-requests="<?php echo $requestsOn ? '1' : '0'; ?>">
<div class="page">
    <header class="header">
        <div class="header__inner">
            <div class="header__brand">
                <div class="header__title"><?php echo htmlspecialchars(STATION_NAME); ?></div>
                <div class="header__subtitle">
                    Curated station library - <?php echo $requestsOn ? 'Request tracks live' : 'Requests currently offline'; ?>
                </div>
            </div>
            <div class="player">
                <button class="player__btn" id="playerToggle" aria-label="Play / Pause">
                    <i class="fa fa-play"></i>
                </button>
                <div class="player__meta">
   		 <div class="player__station" id="playerTrack">Loading</div>
   		 <div class="player__status" id="playerStatus">Stopped</div>
                </div>
                <div class="player__volume-wrap">
                    <i class="fa fa-volume-up player__volume-icon"></i>
                    <input id="playerVolume" class="player__volume" type="range" min="0" max="1" step="0.01" value="0.8">
                </div>
                <audio id="radioAudio" preload="none">
                    <source src="<?php echo htmlspecialchars(STREAM_URL); ?>" type="audio/mpeg">
                </audio>
            </div>
        </div>
    </header>

    <main>
        <div class="main-inner">
            <section class="panel">
                <div class="panel__header">
                    <div>
                        <div class="panel__title">Station Library</div>
                        <div class="panel__subtitle" id="summaryText">Loading songs</div>
                    </div>
                    <div class="search">
                        <span class="search__icon"><i class="fa fa-search"></i></span>
                        <input type="text" id="searchInput" placeholder="Search artist, title, album">
                    </div>
                </div>

                <div class="songs-grid" id="songsGrid">
                    <div class="empty-grid">Loading...</div>
                </div>

                <div class="meta-bar">
                    <div id="resultsInfo"></div>
                    <div class="pagination"><span style="margin-right:50px;">Powered by <a href="https://www.radiodj.ro/" target="_blank" style="color:#9ba3c6;">RadioDJ</a> | Developed by <a href="https://www.streamcodestudio.com/" target="_blank" style="color:#9ba3c6;">StreamCode Studio</a></span>
                        <button id="prevPage">
                            <i class="fa fa-chevron-left"></i> Prev
                        </button>
                        <span id="pageInfo">Page 1 / 1</span>
                        <button id="nextPage">
                            Next <i class="fa fa-chevron-right"></i>
                        </button>
                    </div>
                </div>
            </section>
        </div>
    </main>
</div>

<!-- Request Modal -->
<div class="modal-backdrop" id="requestModal">
    <div class="modal">
        <div class="modal__hero" id="modalHero">
            <div class="modal__hero-inner">
                <div class="modal__hero-artist" id="modalHeroArtist">Artist</div>
                <div class="modal__hero-title" id="modalHeroTitle">Song title</div>
            </div>
            <button class="modal__close-btn" type="button" id="modalClose">
                <i class="fa fa-times"></i>
            </button>
        </div>
        <div class="modal__body">
            <div class="modal__sub">
                Drop your name and (optionally) a dedication. Well line this up in the playlist.
            </div>
            <form id="requestForm">
                <input type="hidden" name="song_id" id="modalSongId">
                <div class="form-grid">
                    <div class="field">
                        <label for="reqName">Your name</label>
                        <input type="text" id="reqName" name="username" placeholder="Your name">
                    </div>
                    <div class="field">
                        <label for="reqMessage">Message / dedication (optional)</label>
                        <textarea id="reqMessage" name="message" placeholder="Who's this for? Any shout-outs?"></textarea>
                    </div>
                </div>
                <div class="modal__footer">
                    <div class="modal__hint">Your IP is logged to prevent abuse.</div>
                    <button class="btn--primary" type="submit">
                        <i class="fa fa-paper-plane"></i> Send request
                    </button>
                </div>
                <div class="alert alert--error" id="reqError" style="display:none;"></div>
                <div class="alert alert--success" id="reqSuccess" style="display:none;"></div>
            </form>
        </div>
    </div>
</div>

<script>
(function () {
    const requestsEnabled = document.body.dataset.requests === '1';

    // STREAM PLAYER
    const audio       = document.getElementById('radioAudio');
    const btn         = document.getElementById('playerToggle');
    const btnIcon     = btn.querySelector('i');
    const vol         = document.getElementById('playerVolume');
    const statusLabel = document.getElementById('playerStatus');

    btn.addEventListener('click', () => {
        if (audio.paused) {
            audio.play().then(() => {
                btnIcon.classList.remove('fa-play');
                btnIcon.classList.add('fa-pause');
                statusLabel.textContent = 'Live stream playing';
            }).catch(() => {
                statusLabel.textContent = 'Unable to start stream';
            });
        } else {
            audio.pause();
            btnIcon.classList.remove('fa-pause');
            btnIcon.classList.add('fa-play');
            statusLabel.textContent = 'Stopped';
        }
    });

    vol.addEventListener('input', () => {
        audio.volume = parseFloat(vol.value);
    });

    // SONG GRID + SEARCH + PAGINATION
    const grid        = document.getElementById('songsGrid');
    const pageInfo    = document.getElementById('pageInfo');
    const resultsInfo = document.getElementById('resultsInfo');
    const summaryText = document.getElementById('summaryText');
    const prevBtn     = document.getElementById('prevPage');
    const nextBtn     = document.getElementById('nextPage');
    const searchInput = document.getElementById('searchInput');

    const state = {
        page: 1,
        totalPages: 1,
        q: ''
    };

    function trimText(text, max = 40) {
        text = text || '';
        return text.length > max ? text.substring(0, max) + '...' : text;
    }

    function formatDuration(sec) {
        sec = parseInt(sec || 0, 10);
        const m = Math.floor(sec / 60);
        const s = sec % 60;
        return m + ':' + (s < 10 ? '0' + s : s);
    }

    function renderSongs(data) {
        state.page       = data.page;
        state.totalPages = data.totalPages;

        if (!data.songs.length) {
            grid.innerHTML = '<div class="empty-grid">No matches in the library. Try a different artist, title or album.</div>';
        } else {
            grid.innerHTML = data.songs.map(song => `
                <article class="song-card"
                    data-song-id="${song.id}"
                    data-song-artist="${song.artist || ''}"
                    data-song-title="${song.title || ''}"
                    data-song-art="${song.art || ''}">
                    <div class="song-card__art">
                        <img src="${song.art || ''}" alt="">
                    </div>
                    <div class="song-card__body">
                        <div class="song-title" title="${song.title || ''}">
                            ${trimText(song.title, 40) || '&nbsp;'}
                        </div>
                        <div class="song-artist" title="${song.artist || ''}">
                            ${trimText(song.artist, 40) || '&nbsp;'}
                        </div>
                        <div class="song-likes">Likes: ${song.likes ?? 0}</div>
                    </div>
                    <div class="song-card__footer">
                        ${
                            requestsEnabled
                                ? '<button class="song-request-btn" type="button"><i class="fa fa-paper-plane"></i><span>Request</span></button>'
                                : '<button class="song-request-btn" type="button" disabled><i class="fa fa-ban"></i><span>Off</span></button>'
                        }
                    </div>
                </article>
            `).join('');
        }

        pageInfo.textContent = `Page ${data.page} / ${data.totalPages}`;
        resultsInfo.textContent = `${data.total.toLocaleString()} tracks  showing page ${data.page}`;
        summaryText.textContent = state.q
            ? `Search results for "${state.q}"`
            : 'Browse the full station library and send instant requests.';
        prevBtn.disabled = state.page <= 1;
        nextBtn.disabled = state.page >= state.totalPages;
    }

    async function loadSongs() {
        grid.innerHTML = '<div class="empty-grid">Loading...</div>';
        const params = new URLSearchParams({
            ajax: 'songs',
            page: state.page.toString(),
            q: state.q
        });
        try {
            const res  = await fetch('?' + params.toString(), {cache: 'no-store'});
            const data = await res.json();
            renderSongs(data);
        } catch (e) {
            grid.innerHTML = '<div class="empty-grid">Error loading songs. Please try again.</div>';
        }
    }

    prevBtn.addEventListener('click', () => {
        if (state.page > 1) {
            state.page--;
            loadSongs();
        }
    });

    nextBtn.addEventListener('click', () => {
        if (state.page < state.totalPages) {
            state.page++;
            loadSongs();
        }
    });

    let searchTimer = null;
    searchInput.addEventListener('input', () => {
        state.q = searchInput.value.trim();
        state.page = 1;
        clearTimeout(searchTimer);
        searchTimer = setTimeout(loadSongs, 260);
    });

    // Delegate request clicks
    grid.addEventListener('click', (e) => {
        const btn = e.target.closest('.song-request-btn');
        if (!btn || !requestsEnabled || btn.disabled) return;
        const card = btn.closest('.song-card');
        if (!card) return;

        openRequestModal({
            id: card.dataset.songId,
            title: card.dataset.songTitle || '',
            artist: card.dataset.songArtist || '',
            art: card.dataset.songArt || ''
        });
    });

    loadSongs();

    // REQUEST MODAL (Style 1)
    const modal           = document.getElementById('requestModal');
    const modalClose      = document.getElementById('modalClose');
    const modalHero       = document.getElementById('modalHero');
    const modalHeroArtist = document.getElementById('modalHeroArtist');
    const modalHeroTitle  = document.getElementById('modalHeroTitle');
    const modalSongId     = document.getElementById('modalSongId');
    const requestForm     = document.getElementById('requestForm');
    const reqError        = document.getElementById('reqError');
    const reqSuccess      = document.getElementById('reqSuccess');

    function openRequestModal(song) {
        modalSongId.value = song.id;
        modalHeroArtist.textContent = song.artist || 'Unknown artist';
        modalHeroTitle.textContent  = song.title || 'Unknown title';
        if (song.art) {
            modalHero.style.backgroundImage = 'url(' + song.art + ')';
        } else {
            modalHero.style.backgroundImage = 'none';
        }

        reqError.style.display   = 'none';
        reqSuccess.style.display = 'none';

        requestForm.reset();
        modalSongId.value = song.id;

        modal.classList.add('is-visible');
    }

    function closeRequestModal() {
        modal.classList.remove('is-visible');
    }

    modalClose.addEventListener('click', closeRequestModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeRequestModal();
        }
    });

    requestForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        reqError.style.display   = 'none';
        reqSuccess.style.display = 'none';

        const formData = new FormData(requestForm);
        formData.append('action', 'request');

        try {
            const res  = await fetch('', {
                method: 'POST',
                body: formData
            });
            const raw = await res.text();
            let data;
            try {
                data = JSON.parse(raw);
            } catch (parseErr) {
                console.error('Non-JSON response from server:', raw);
                reqError.textContent = 'Server error (invalid response). Please check PHP error log.';
                reqError.style.display = 'block';
                return;
            }

            if (!data.ok) {
                reqError.textContent = data.error || 'Something went wrong.';
                reqError.style.display = 'block';
            } else {
                reqSuccess.textContent = data.message || 'Request sent!';
                reqSuccess.style.display = 'block';
                setTimeout(closeRequestModal, 2500);
            }
        } catch (err) {
            console.error('Fetch failed:', err);
            reqError.textContent = 'Network error. Please try again.';
            reqError.style.display = 'block';
        }
    });
async function loadNowPlaying() {
    try {
        const res = await fetch("nowplaying.live.php?t=" + Date.now(), {
            cache: "no-store"
        });

        const data = await res.json();

        const artist = data.artist || "";
        const title  = data.title || "";

        const trackEl = document.getElementById("playerTrack");

        if (!trackEl) return;

        if (artist && title) {
            trackEl.textContent = `${artist} - ${title}`;
        } else if (title) {
            trackEl.textContent = title;
        } else {
            trackEl.textContent = "Live Stream";
        }
    } catch (err) {
        document.getElementById("playerTrack").textContent = "Live Stream";
    }
}

loadNowPlaying();
setInterval(loadNowPlaying, 10000);
})();
</script>
</body>
</html>
