<?php
declare(strict_types=1);

// --------------------------------------------------
// BASIC SETUP
// --------------------------------------------------
date_default_timezone_set('Europe/London');

// --------------------------------------------------
// CONFIG: EDIT THESE
// --------------------------------------------------

// Database connection
define('DB_HOST', 'IP ADDRESS:3306');
define('DB_NAME', 'radiodj-database');
define('DB_USER', 'radiodj-user');
define('DB_PASS', 'radiodj-password');

// Station / stream details
define('STATION_NAME',  'Your station name');
define('STREAM_URL',    'https://uk2-pn.mixstream.net/PORTNUM/stream'); // <- your stream URL

// How many songs per page in the list
define('SONGS_PER_PAGE', 25);

// Requests system
define('REQUESTS_ALLOW', 'y'); // 'y' = enabled, 'n' = disabled

// --------------------------------------------------
// PDO CONNECTION ($pdo_scs as requested)
// --------------------------------------------------
$dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';

try {
    $pdo_scs = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo "Database connection failed.";
    exit;
}

// --------------------------------------------------
// AJAX: SONG LIST (LIVE SEARCH + PAGINATION)
// --------------------------------------------------
if (isset($_GET['ajax']) && $_GET['ajax'] === 'songs') {
    $page    = max(1, (int)($_GET['page'] ?? 1));
    $perPage = SONGS_PER_PAGE;
    $offset  = ($page - 1) * $perPage;
    $q       = trim((string)($_GET['q'] ?? ''));

    $where  = '';
    $params = [];

    if ($q !== '') {
        $where = "WHERE (artist LIKE :q OR title LIKE :q OR album LIKE :q)";
        $params[':q'] = '%' . $q . '%';
    }

    // Total count
    $sqlCount = "SELECT COUNT(*) FROM songs $where";
    $stmtCount = $pdo_scs->prepare($sqlCount);
    if ($q !== '') {
        $stmtCount->bindValue(':q', '%' . $q . '%', PDO::PARAM_STR);
    }
    $stmtCount->execute();
    $total = (int)$stmtCount->fetchColumn();
    $totalPages = max(1, (int)ceil($total / $perPage));

    // Paged rows
    $sql = "
        SELECT
            id,
            artist,
            title,
            album,
            duration
        FROM songs
        $where
        ORDER BY artist ASC, title ASC
        LIMIT :limit OFFSET :offset
    ";
    $stmt = $pdo_scs->prepare($sql);

    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v, PDO::PARAM_STR);
    }
    $stmt->bindValue(':limit',  $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset,  PDO::PARAM_INT);
    $stmt->execute();

    $songs = $stmt->fetchAll();

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'songs'      => $songs,
        'page'       => $page,
        'totalPages' => $totalPages,
        'total'      => $total,
    ]);
    exit;
}

// --------------------------------------------------
// AJAX: REQUEST TRACK
// --------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'request') {
    header('Content-Type: application/json; charset=utf-8');

    if (REQUESTS_ALLOW !== 'y') {
        echo json_encode([
            'ok'    => false,
            'error' => 'Requests are currently disabled.',
        ]);
        exit;
    }

    $songId   = (int)($_POST['song_id'] ?? 0);
    $username = trim((string)($_POST['username'] ?? ''));
    $message  = trim((string)($_POST['message'] ?? ''));

    if ($songId < 1 || $username === '') {
        echo json_encode([
            'ok'    => false,
            'error' => 'Please enter your name to send a request.',
        ]);
        exit;
    }

    // Optional: ensure song exists
    $chk = $pdo_scs->prepare("SELECT id FROM songs WHERE id = :id LIMIT 1");
    $chk->execute([':id' => $songId]);
    if (!$chk->fetch()) {
        echo json_encode([
            'ok'    => false,
            'error' => 'Song not found.',
        ]);
        exit;
    }

    // Insert into your requests table
    // Adjust column names to match your DB if needed
$stmt = $pdo_scs->prepare("
    INSERT INTO requests
        (songID, username, message, userIP, requested)
    VALUES
        (:songID, :username, :message, :userIP, NOW())
");

$stmt->execute([
    ':songID'      => $songId,
    ':username'=> mb_substr($username, 0, 100),
    ':message' => mb_substr($message, 0, 500),
    ':userIP'  => $_SERVER['REMOTE_ADDR'] ?? '',
]);
    echo json_encode([
        'ok'      => true,
        'message' => 'Your request has been sent to the playlist!',
    ]);
    exit;
}

// --------------------------------------------------
// MAIN PAGE OUTPUT
// --------------------------------------------------
$requestsOn = (REQUESTS_ALLOW === 'y');if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    echo json_encode(['ok' => false, 'error' => 'Handler not reached']);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Requests | <?php echo htmlspecialchars(STATION_NAME); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- Font Awesome (icons) -->
<link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    <style>
        :root {
            --bg: #050914;
            --bg-alt: #0b101f;
            --bg-elevated: #111729;
            --accent: #852c46;
            --accent-soft: #2b1b3f;
            --accent-soft-2: #1c273e;
            --text: #f5f7ff;
            --text-muted: #9ba3c6;
            --border-subtle: #252a3d;
        }

        * {
            box-sizing: border-box;
        }

        body {
            margin: 0;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: radial-gradient(circle at top, #141b33 0, var(--bg) 55%);
            color: var(--text);
        }

        .page {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }

        .header {
            position: sticky;
            top: 0;
            z-index: 20;
            backdrop-filter: blur(14px);
            background: linear-gradient(to right, rgba(5, 9, 20, 0.96), rgba(10, 12, 24, 0.96));
            border-bottom: 1px solid var(--border-subtle);
        }

        .header__inner {
            max-width: 1100px;
            margin: 0 auto;
            padding: 12px 16px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 16px;
        }

        .header__brand {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }

        .header__title {
            font-weight: 700;
            letter-spacing: .08em;
            font-size: 0.95rem;
            text-transform: uppercase;
        }

        .header__subtitle {
            font-size: 0.78rem;
            color: var(--text-muted);
        }

        .player {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 8px 12px;
            border-radius: 999px;
            background: radial-gradient(circle at top, var(--accent-soft), transparent 70%);
            border: 1px solid rgba(255, 75, 110, 0.4);
            box-shadow: 0 0 25px rgba(255, 75, 110, 0.3);
        }

        .player__btn {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            border: none;
            cursor: pointer;
            background: var(--accent);
            color: #fff;
            font-size: 14px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .player__btn i {
            pointer-events: none;
        }

        .player__btn:active {
            transform: scale(0.96);
        }

        .player__meta {
            display: flex;
            flex-direction: column;
            gap: 2px;
            min-width: 0;
        }

        .player__station {
            font-size: 0.8rem;
            font-weight: 600;
        }

        .player__status {
            font-size: 0.75rem;
            color: var(--text-muted);
        }

        .player__track {
            font-size: 0.75rem;
            color: var(--text-muted);
            white-space: nowrap;
            text-overflow: ellipsis;
            overflow: hidden;
            max-width: 260px;
        }

        .player__volume-wrap {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .player__volume {
            width: 120px;
        }

        .player__volume-icon {
            font-size: 0.8rem;
            color: var(--text-muted);
        }

        main {
            flex: 1;
        }

        .main-inner {
            max-width: 1100px;
            margin: 18px auto 32px;
            padding: 0 16px 24px;
        }

        .panel {
            background: linear-gradient(145deg, rgba(17, 23, 41, 0.96), rgba(12, 16, 27, 0.98));
            border-radius: 18px;
            border: 1px solid var(--border-subtle);
            box-shadow: 0 26px 60px rgba(0, 0, 0, 0.6);
            padding: 18px 18px 8px;
        }

        .panel__header {
            display: flex;
            flex-wrap: wrap;
            gap: 12px;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
        }

        .panel__title {
            font-size: 1.05rem;
            font-weight: 600;
        }

        .panel__subtitle {
            font-size: 0.8rem;
            color: var(--text-muted);
        }

        .search {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 6px 10px;
            border-radius: 999px;
            background: var(--bg-alt);
            border: 1px solid var(--border-subtle);
        }

        .search input {
            background: transparent;
            border: none;
            outline: none;
            color: var(--text);
            font-size: 0.85rem;
            width: 180px;
        }

        .search input::placeholder {
            color: var(--text-muted);
        }

        .search__icon {
            font-size: 0.9rem;
            color: var(--text-muted);
        }

        .table-wrap {
            border-radius: 12px;
            overflow: hidden;
            border: 1px solid var(--border-subtle);
            background: var(--bg-alt);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.85rem;
        }

        thead {
            background: linear-gradient(to right, var(--accent-soft-2), var(--bg-alt));
        }

        th, td {
            padding: 8px 10px;
            border-bottom: 1px solid #1c2236;
        }

        th {
            text-align: left;
            font-weight: 500;
            color: var(--text-muted);
            font-size: 0.78rem;
        }

        tbody tr:nth-child(even) {
            background: rgba(10, 15, 30, 0.8);
        }

        tbody tr:hover {
            background: rgba(36, 46, 85, 0.9);
        }

        .col-artist { width: 28%; }
        .col-title  { width: 34%; }
        .col-album  { width: 24%; }
        .col-duration {
            width: 7%;
            white-space: nowrap;
        }
        .col-actions {
            width: 7%;
            text-align: right;
        }

        .btn {
            border: none;
            border-radius: 999px;
            padding: 4px 10px;
            font-size: 0.78rem;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn i {
            font-size: 0.8rem;
        }

        .btn--request {
            background: linear-gradient(135deg, var(--accent), #af736a);
            color: #fff;
            box-shadow: 0 0 16px rgba(255, 138, 119, 0.4);
        }

        .btn--request.is-disabled {
            opacity: 0.4;
            cursor: default;
            box-shadow: none;
        }

        .meta-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 10px;
            margin: 10px 2px 8px;
            font-size: 0.78rem;
            color: var(--text-muted);
        }

        .pagination {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .pagination button {
            border-radius: 999px;
            border: 1px solid var(--border-subtle);
            background: var(--bg-alt);
            color: var(--text);
            font-size: 0.76rem;
            padding: 4px 10px;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 4px;
        }

        .pagination button[disabled] {
            opacity: .4;
            cursor: default;
        }

        .pagination__info {
            font-size: 0.76rem;
        }

        .empty-row {
            text-align: center;
            padding: 20px 10px;
            color: var(--text-muted);
        }

        /* MODAL */
        .modal-backdrop {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.75);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 40;
        }

        .modal-backdrop.is-visible {
            display: flex;
        }

        .modal {
            width: 100%;
            max-width: 420px;
            background: linear-gradient(145deg, var(--bg-elevated), #141c34);
            border-radius: 18px;
            border: 1px solid var(--accent-soft);
            box-shadow: 0 26px 70px rgba(0, 0, 0, 0.85);
            padding: 16px 16px 14px;
        }

        .modal__header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 8px;
        }

        .modal__title {
            font-size: 0.95rem;
            font-weight: 600;
        }

        .modal__close {
            border: none;
            background: transparent;
            color: var(--text-muted);
            cursor: pointer;
            font-size: 1.1rem;
        }

        .modal__sub {
            font-size: 0.78rem;
            color: var(--text-muted);
            margin-bottom: 8px;
        }

        .modal__song {
            font-size: 0.8rem;
            background: radial-gradient(circle at top left, var(--accent-soft), transparent 70%);
            border-radius: 12px;
            padding: 8px 9px;
            margin-bottom: 8px;
        }

        .modal__song span {
            display: block;
        }

        .modal__song-title {
            font-weight: 600;
        }

        .modal__song-artist {
            font-size: 0.76rem;
            color: var(--text-muted);
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 8px;
            margin-bottom: 8px;
        }

        .field label {
            display: block;
            font-size: 0.75rem;
            margin-bottom: 3px;
            color: var(--text-muted);
        }

        .field input,
        .field textarea {
            width: 100%;
            border-radius: 9px;
            padding: 7px 8px;
            border: 1px solid var(--border-subtle);
            background: #10182b;
            color: var(--text);
            font-size: 0.8rem;
            resize: vertical;
            min-height: 36px;
        }

        .field textarea {
            min-height: 60px;
        }

        .field input::placeholder,
        .field textarea::placeholder {
            color: var(--text-muted);
        }

        .modal__footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 6px;
        }

        .modal__hint {
            font-size: 0.72rem;
            color: var(--text-muted);
        }

        .btn--primary {
            background: linear-gradient(135deg, var(--accent), #ff8a77);
            color: #fff;
            border: none;
            border-radius: 999px;
            padding: 6px 14px;
            font-size: 0.8rem;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn--primary i {
            font-size: 0.8rem;
        }

        .alert {
            font-size: 0.78rem;
            margin-top: 4px;
        }

        .alert--error {
            color: #ffb3b3;
        }

        .alert--success {
            color: #a8ffcb;
        }

        @media (max-width: 640px) {
            .header__inner {
                flex-direction: column;
                align-items: flex-start;
            }
            .player {
                width: 100%;
                justify-content: space-between;
            }
            .player__volume {
                width: 90px;
            }
            .search input {
                width: 130px;
            }
            .panel {
                padding: 14px 12px 8px;
            }
            th:nth-child(3), td:nth-child(3) {
                display: none; /* hide album on tiny screens */
            }
            .player__track {
                max-width: 180px;
            }
        }
    </style>
</head>
<body data-requests="<?php echo $requestsOn ? '1' : '0'; ?>">
<div class="page">
    <header class="header">
        <div class="header__inner">
            <div class="header__brand">
                <div class="header__title"><?php echo htmlspecialchars(STATION_NAME); ?></div>
                <div class="header__subtitle">
                    Browse the library - <?php echo $requestsOn ? 'Request tracks live' : 'Requests currently offline'; ?>
                </div>
            </div>
            <div class="player">
                <button class="player__btn" id="playerToggle" aria-label="Play / Pause">
                    <i class="fa fa-play"></i>
                </button>
                <div class="player__meta">
                    <div class="player__station">Live Stream</div>
                    <div class="player__status" id="playerStatus">Stopped</div>
                    <div class="player__track" id="playerTrack">Station library</div>
                </div>
                <div class="player__volume-wrap">
                    <i class="fa fa-volume-up player__volume-icon"></i>
                    <input id="playerVolume" class="player__volume" type="range" min="0" max="1" step="0.01" value="0.8">
                </div>
                <audio id="radioAudio" preload="none">
                    <source src="<?php echo htmlspecialchars(STREAM_URL); ?>" type="audio/mpeg">
                </audio>
            </div>
        </div>
    </header>

    <main>
        <div class="main-inner">
            <section class="panel">
                <div class="panel__header">
                    <div>
                        <div class="panel__title">Song Library</div>
                        <div class="panel__subtitle" id="summaryText">
                            Loading songs...
                        </div>
                    </div>
                    <div class="search">
                        <span class="search__icon"><i class="fa fa-search"></i></span>
                        <input type="text" id="searchInput" placeholder="Search artist, title, album">
                    </div>
                </div>

                <div class="table-wrap">
                    <table>
                        <thead>
                        <tr>
                            <th class="col-artist">Artist</th>
                            <th class="col-title">Title</th>
                            <th class="col-album">Album</th>
                            <th class="col-duration">Time</th>
                            <th class="col-actions"><?php echo $requestsOn ? '' : 'Requests'; ?></th>
                        </tr>
                        </thead>
                        <tbody id="songsBody">
                        <tr><td colspan="5" class="empty-row">Loading...</td></tr>
                        </tbody>
                    </table>
                </div>

                <div class="meta-bar">
                    <div id="resultsInfo"> </div>
                    <div class="pagination"><span style="margin-right:50px;">Powered by <a href="https://www.radiodj.ro/" target="_blank" style="color:#9ba3c6;">RadioDJ</a> | Developed by <a href="https://www.streamcodestudio.com/" target="_blank" style="color:#9ba3c6;">StreamCode Studio</a></span>  
                        <button id="prevPage">
                            <i class="fa fa-chevron-left"></i> Prev
                        </button>
                        <span class="pagination__info" id="pageInfo">Page 1 / 1</span>
                        <button id="nextPage">
                            Next <i class="fa fa-chevron-right"></i>
                        </button>
                    </div>
                </div>
            </section>
        </div>
    </main>
</div>

<!-- Request Modal -->
<div class="modal-backdrop" id="requestModal">
    <div class="modal">
        <div class="modal__header">
            <div class="modal__title"><i class="fa fa-paper-plane"></i> Request this track</div>
            <button class="modal__close" type="button" id="modalClose">&times;</button>
        </div>
        <div class="modal__sub">Drop your name and (optionally) a message or dedication.</div>
        <div class="modal__song">
            <span class="modal__song-title" id="modalSongTitle">Song title</span>
            <span class="modal__song-artist" id="modalSongArtist">Artist</span>
        </div>
        <form id="requestForm">
            <input type="hidden" name="song_id" id="modalSongId">
            <div class="form-grid">
                <div class="field">
                    <label for="reqName">Your name</label>
                    <input type="text" id="reqName" name="username" placeholder="Your name">
                </div>
                <div class="field">
                    <label for="reqMessage">Message / dedication (optional)</label>
                    <textarea id="reqMessage" name="message" placeholder="Who's this for? Any shout-outs?"></textarea>
                </div>
            </div>
            <div class="modal__footer">
                <button class="btn--primary" type="submit">
                    <i class="fa fa-paper-plane"></i> Send request
                </button>
            </div>
            <div class="alert alert--error" id="reqError" style="display:none;"></div>
            <div class="alert alert--success" id="reqSuccess" style="display:none;"></div>
        </form>
    </div>
</div>

<script>
(function () {
    const requestsEnabled = document.body.dataset.requests === '1';

    // --------------------------------------------------
    // STREAM PLAYER
    // --------------------------------------------------
    const audio       = document.getElementById('radioAudio');
    const btn         = document.getElementById('playerToggle');
    const btnIcon     = btn.querySelector('i');
    const vol         = document.getElementById('playerVolume');
    const statusLabel = document.getElementById('playerStatus');
    const playerTrack = document.getElementById('playerTrack');

    btn.addEventListener('click', () => {
        if (audio.paused) {
            audio.play().then(() => {
                btnIcon.classList.remove('fa-play');
                btnIcon.classList.add('fa-pause');
                statusLabel.textContent = 'Live stream playing';
            }).catch(() => {
                statusLabel.textContent = 'Unable to start stream';
            });
        } else {
            audio.pause();
            btnIcon.classList.remove('fa-pause');
            btnIcon.classList.add('fa-play');
            statusLabel.textContent = 'Stopped';
        }
    });

    vol.addEventListener('input', () => {
        audio.volume = parseFloat(vol.value);
    });

    // --------------------------------------------------
    // SONG LIST + LIVE SEARCH + PAGINATION
    // --------------------------------------------------
    const tbody       = document.getElementById('songsBody');
    const pageInfo    = document.getElementById('pageInfo');
    const resultsInfo = document.getElementById('resultsInfo');
    const summaryText = document.getElementById('summaryText');
    const prevBtn     = document.getElementById('prevPage');
    const nextBtn     = document.getElementById('nextPage');
    const searchInput = document.getElementById('searchInput');

    const state = {
        page: 1,
        totalPages: 1,
        q: ''
    };

    function formatDuration(sec) {
        sec = parseInt(sec || 0, 10);
        const m = Math.floor(sec / 60);
        const s = sec % 60;
        return m + ':' + (s < 10 ? '0' + s : s);
    }

    function renderSongs(data) {
        state.page       = data.page;
        state.totalPages = data.totalPages;

        if (!data.songs.length) {
            tbody.innerHTML = '<tr><td colspan="5" class="empty-row">No matches in the library. Try a different artist, title or album.</td></tr>';
        } else {
            tbody.innerHTML = data.songs.map(song => `
                <tr data-song-id="${song.id}"
                    data-song-artist="${song.artist || ''}"
                    data-song-title="${song.title || ''}">
                    <td class="col-artist">${song.artist || '&nbsp;'}</td>
                    <td class="col-title">${song.title || '&nbsp;'}</td>
                    <td class="col-album">${song.album || '&nbsp;'}</td>
                    <td class="col-duration">${formatDuration(song.duration)}</td>
                    <td class="col-actions">
                        ${requestsEnabled
                            ? '<button class="btn btn--request" type="button"><i class="fa fa-paper-plane"></i><span>Request</span></button>'
                            : '<button class="btn btn--request is-disabled" type="button" disabled><i class="fa fa-ban"></i><span>Off</span></button>'}
                    </td>
                </tr>
            `).join('');
        }

        pageInfo.textContent = `Page ${data.page} / ${data.totalPages}`;
        resultsInfo.textContent = `${data.total.toLocaleString()} tracks - showing page ${data.page}`;
        summaryText.textContent = state.q
            ? `Search results for "${state.q}"`
            : 'Browse the full station library and send instant requests.';

        prevBtn.disabled = state.page <= 1;
        nextBtn.disabled = state.page >= state.totalPages;
    }

    async function loadSongs() {
        tbody.innerHTML = '<tr><td colspan="5" class="empty-row">Loading...</td></tr>';
        const params = new URLSearchParams({
            ajax: 'songs',
            page: state.page.toString(),
            q: state.q
        });
        try {
            const res  = await fetch('?' + params.toString(), {cache: 'no-store'});
            const data = await res.json();
            renderSongs(data);
        } catch (e) {
            tbody.innerHTML = '<tr><td colspan="5" class="empty-row">Error loading songs. Please try again.</td></tr>';
        }
    }

    prevBtn.addEventListener('click', () => {
        if (state.page > 1) {
            state.page--;
            loadSongs();
        }
    });

    nextBtn.addEventListener('click', () => {
        if (state.page < state.totalPages) {
            state.page++;
            loadSongs();
        }
    });

    let searchTimer = null;
    searchInput.addEventListener('input', () => {
        state.q = searchInput.value.trim();
        state.page = 1;
        clearTimeout(searchTimer);
        searchTimer = setTimeout(loadSongs, 260);
    });

    // delegate click for "Request" buttons
    tbody.addEventListener('click', (e) => {
        const btn = e.target.closest('.btn--request');
        if (!btn || !requestsEnabled || btn.disabled) return;
        const row = btn.closest('tr');
        if (!row) return;
        const song = {
            id: row.dataset.songId,
            title: row.dataset.songTitle || '',
            artist: row.dataset.songArtist || ''
        };
        openRequestModal(song);
    });

    loadSongs();

    // --------------------------------------------------
    // REQUEST MODAL LOGIC
    // --------------------------------------------------
    const modal           = document.getElementById('requestModal');
    const modalClose      = document.getElementById('modalClose');
    const modalSongTitle  = document.getElementById('modalSongTitle');
    const modalSongArtist = document.getElementById('modalSongArtist');
    const modalSongId     = document.getElementById('modalSongId');
    const requestForm     = document.getElementById('requestForm');
    const reqError        = document.getElementById('reqError');
    const reqSuccess      = document.getElementById('reqSuccess');

    function setPlayerTrackFromSong(song) {
        const text = (song.artist && song.title)
            ? `${song.artist}  ${song.title}`
            : (song.title || song.artist || 'Station library');
        playerTrack.textContent = text;
    }

    function openRequestModal(song) {
        modalSongId.value           = song.id;
        modalSongTitle.textContent  = song.title || 'Unknown title';
        modalSongArtist.textContent = song.artist || 'Unknown artist';

        reqError.style.display   = 'none';
        reqSuccess.style.display = 'none';

        // reset form but keep the hidden song id
        requestForm.reset();
        modalSongId.value = song.id;

        modal.classList.add('is-visible');
    }

    function closeRequestModal() {
        modal.classList.remove('is-visible');
    }

    modalClose.addEventListener('click', closeRequestModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            closeRequestModal();
        }
    });

requestForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    reqError.style.display   = 'none';
    reqSuccess.style.display = 'none';

    const formData = new FormData(requestForm);
    formData.append('action', 'request');

    try {
        const res  = await fetch('', {
            method: 'POST',
            body: formData
        });

        const raw = await res.text();          // <-- grab raw response
        let data;

        try {
            data = JSON.parse(raw);           // <-- try to parse JSON
        } catch (parseErr) {
            console.error('Non-JSON response from server:', raw);
            reqError.textContent = 'Server error (invalid response). Check console / PHP error log.';
            reqError.style.display = 'block';
            return;
        }

        if (!data.ok) {
            reqError.textContent = data.error || 'Something went wrong.';
            reqError.style.display = 'block';
        } else {
            reqSuccess.textContent = data.message || 'Request sent!';
            reqSuccess.style.display = 'block';
            setTimeout(closeRequestModal, 4200);
        }
    } catch (err) {
        console.error('Fetch failed:', err);
        reqError.textContent = 'Network error. Please try again.';
        reqError.style.display = 'block';
    }
});
    // --------------------------------------------------
    // NOW PLAYING from RadioDJ history table (INSIDE IIFE)
    // --------------------------------------------------
    async function loadNowPlaying() {
        try {
            const res = await fetch("nowplaying.live.php?t=" + Date.now(), {
                cache: "no-store"
            });
            const data = await res.json();

            const artist = data.artist || "";
            const title  = data.title || "";

            if (artist && title) {
                playerTrack.textContent = artist + " - " + title;
            } else if (title) {
                playerTrack.textContent = title;
            } else {
                playerTrack.textContent = "Live stream";
            }

        } catch (e) {
            playerTrack.textContent = "Live stream";
        }
    }

    loadNowPlaying();
    setInterval(loadNowPlaying, 15000);
})();

</script>
</body>
</html>
