if (!window.CB_PLAYER_INITIALIZED) {
    window.CB_PLAYER_INITIALIZED = true;
(function () {
    const NOWPLAYING_URL = "nowplaying.php";

    // --- ELEMENTS ---
    const audio        = document.getElementById("cb-audio");
    const titleEl      = document.getElementById("cb-title");
    const artistEl     = document.getElementById("cb-artist");
    const artImg       = document.getElementById("cb-art-img");
    const playBtn      = document.getElementById("cb-play");
    const playIcon     = document.getElementById("cb-play-icon");
    const volumeRange  = document.getElementById("cb-volume-range");
    const muteBtn      = document.getElementById("cb-mute");
    const requestedEl  = document.getElementById("cb-requested-by"); // optional
    const bufferingEl  = document.getElementById("cb-buffering");    // optional

    // If key elements are missing, bail quietly
    if (!audio || !titleEl || !artistEl || !artImg || !playBtn || !playIcon) {
        console.warn("[CB Player] Missing core DOM elements, aborting init.");
        return;
    }

    let lastVolume = 0.7;
    let firstStreamSet = false;

    audio.volume = lastVolume;

    // ------------------------------------------
    // Small helpers
    // ------------------------------------------

    function setPlayIcon(playing) {
        playIcon.innerHTML = playing
            ? '<path d="M6 5h4v14H6zM14 5h4v14h-4z"/>'
            : '<path d="M8 5v14l11-7z"/>';
    }

    function showBuffering(show) {
        if (!bufferingEl) return;
        bufferingEl.style.display = show ? "inline-flex" : "none";
    }

    function fadeOutIn(elements, updateFn, duration = 200) {
        const els = Array.isArray(elements) ? elements : [elements];

        els.forEach(el => { if (el) el.style.transition = `opacity ${duration}ms`; });
        els.forEach(el => { if (el) el.style.opacity = "0.3"; });

        setTimeout(() => {
            updateFn();
            els.forEach(el => { if (el) el.style.opacity = "1"; });
        }, duration);
    }

    // ------------------------------------------
    // PLAY / PAUSE
    // ------------------------------------------

    playBtn.addEventListener("click", () => {
        if (!audio.src) {
            // If stream URL not set yet, fetch metadata first
            updateNowPlaying().then(() => {
                audio.play().then(() => setPlayIcon(true))
                    .catch(err => console.warn("[CB Player] Autoplay blocked:", err));
            });
            return;
        }

        if (audio.paused) {
            audio.play().then(() => setPlayIcon(true))
                .catch(err => console.warn("[CB Player] Play error:", err));
        } else {
            audio.pause();
            setPlayIcon(false);
        }
    });

    audio.addEventListener("playing", () => {
        setPlayIcon(true);
        showBuffering(false);
    });

    audio.addEventListener("pause", () => {
        setPlayIcon(false);
    });

    audio.addEventListener("waiting", () => {
        showBuffering(true);
    });

    audio.addEventListener("stalled", () => {
        showBuffering(true);
    });

    audio.addEventListener("canplay", () => {
        showBuffering(false);
    });

    audio.addEventListener("error", () => {
        console.warn("[CB Player] Audio error");
        showBuffering(false);
    });

    // ------------------------------------------
    // VOLUME
    // ------------------------------------------

    if (volumeRange) {
        volumeRange.addEventListener("input", function () {
            const v = parseFloat(this.value);
            audio.volume = isNaN(v) ? audio.volume : v;
            lastVolume = audio.volume;
        });
    }

    if (muteBtn) {
        muteBtn.addEventListener("click", function () {
            if (audio.volume > 0) {
                lastVolume = audio.volume;
                audio.volume = 0;
                if (volumeRange) volumeRange.value = "0";
            } else {
                audio.volume = lastVolume || 0.7;
                if (volumeRange) volumeRange.value = String(audio.volume);
            }
        });
    }

    // ------------------------------------------
    // NOW PLAYING UPDATES
    // ------------------------------------------

    async function updateNowPlaying() {
        try {
            const res = await fetch(NOWPLAYING_URL + "?t=" + Date.now(), {
                cache: "no-store"
            });

            if (!res.ok) {
                console.warn("[CB Player] nowplaying.php HTTP error:", res.status);
                return;
            }

            const data = await res.json();

            fadeOutIn(
                [artImg, titleEl, artistEl, requestedEl],
                () => {
                    if (data.title)   titleEl.textContent  = data.title;
                    if (data.artist)  artistEl.textContent = data.artist;
                    if (data.artwork) artImg.src           = data.artwork;

                    if (requestedEl) {
                        if (data.requestedBy || data.requestMsg) {
                            requestedEl.style.display = "block";
				let html = "";

				if (data.requestedBy) {
					html += `<strong>${data.requestedBy}</strong>`;
				}
				if (data.requestMsg) {
					html += `<br><span>${data.requestMsg}</span>`;
				}
				requestedEl.innerHTML = html;
                        } else {
                            requestedEl.style.display = "none";
                            requestedEl.textContent   = "";
                        }
                    }
                },
                200
            );

            if (data.stream && typeof data.stream === "string") {
                if (!firstStreamSet || audio.src !== data.stream) {
                    audio.src = data.stream;
                    firstStreamSet = true;
                    console.log("[CB Player] Stream set to:", data.stream);
                }
            }

        } catch (err) {
            console.error("[CB Player] Now playing fetch failed:", err);
        }
    }

    // Initial load + periodic refresh
updateNowPlaying().then(() => {
    // Try to autoplay once the stream URL is set
    if (audio.src) {
        audio.play()
            .then(() => setPlayIcon(true))
            .catch(err => {
                console.warn("[CB Player] Autoplay was blocked:", err);
            });
    }
});
setInterval(updateNowPlaying, 10000);
})();}
